<?php

namespace App\Http\Controllers\Api;

use App\Models\Order;
use App\Models\State;
use App\Models\Wallet;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Http\Controllers\Controller;
use Carbon\Carbon;
use App\User;
use Illuminate\Support\Facades\Hash;

class AuthController extends Controller
{
    /**
     * Create user
     *
     * @param  [string] name
     * @param  [string] email
     * @param  [string] password
     * @param  [string] password_confirmation
     * @return [string] message
     */
    public function signup(Request $request)
    {
        $request->validate([
            'name' => 'required|string',
            'email' => 'required|string|email|unique:users',
            'password' => 'required|string|confirmed'
        ]);
        $user = new User([
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'state_id' => $request->state_id,
            'city' => $request->city,
            'pincode' => $request->pincode,
            'address' => $request->address,
            'role_id' => "2",
            'password' => bcrypt($request->password)
        ]);
        $user->save();
        return response()->json([
            'message' => 'Successfully created user!', 'user' => $user, 'success' => true, 'statusCode' => 200
        ], 201);
    }

    /**
     * Login user and create token
     *
     * @param  [string] email
     * @param  [string] password
     * @param  [boolean] remember_me
     * @return [string] access_token
     * @return [string] token_type
     * @return [string] expires_at
     */
	  public function socialLogin(Request $request)
    {
        $users = User::where(['email' => $request->email])->first();
        if ($users) {
           // $user = $request->user();
            $tokenResult = $users->createToken('Personal Access Token');
            $token = $tokenResult->token;
            $token->save();
            return response()->json([
                'success' => true, 'statusCode' => 200,
                'user' => $users,
                'message' => 'Login Successful',
                'access_token' => $tokenResult->accessToken,
                'token_type' => 'Bearer',
                'expires_at' => Carbon::parse(
                    $tokenResult->token->expires_at
                )->toDateTimeString()
            ]);
        } else {

            $user = new User([
                'name' => $request->name,
                'email' => $request->email,
                'provider_id' => $request->provider_id,
                'provider' => $request->provider,
                'role_id' => "2",
                'password'=>bcrypt($request->provider_id),

            ]);
            $user->save();

            
            $tokenResult = $user->createToken('Personal Access Token');
            $token = $tokenResult->token;
            $token->save();
            return response()->json([
                'success' => true, 'statusCode' => 200,
                'user' => $user,
                'message' => 'Login Successful',
                'access_token' => $tokenResult->accessToken,
                'token_type' => 'Bearer',
                'expires_at' => Carbon::parse(
                    $tokenResult->token->expires_at
                )->toDateTimeString()
            ]);
        }
    }
    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|string|email',
            'password' => 'required|string',
            'remember_me' => 'boolean'
        ]);
        $credentials = request(['email', 'password']);
        if (!Auth::attempt($credentials))
            return response()->json([
                'message' => 'Unauthorized'
            ], 401);
        $user = $request->user();
        $tokenResult = $user->createToken('Personal Access Token');
        $token = $tokenResult->token;
        if ($request->remember_me)
            $token->expires_at = Carbon::now()->addWeeks(1);
        $token->save();
        return response()->json([
            'user' => $user,
            'message' => 'Login Successful',
            'access_token' => $tokenResult->accessToken,
            'token_type' => 'Bearer',
            'success' => true, 'statusCode' => 200,
            'expires_at' => Carbon::parse(
                $tokenResult->token->expires_at
            )->toDateTimeString()
        ]);
    }

    /**
     * Logout user (Revoke the token)
     *
     * @return [string] message
     */
    public function logout(Request $request)
    {
        $request->user()->token()->revoke();
        return response()->json([
            'message' => 'Successfully logged out', 'success' => true, 'statusCode' => 200
        ]);
    }

    /**
     * Get the authenticated User
     *
     * @return [json] user object
     */
    //my profile
    public function user(Request $request)
    {
        $state = State::find(Auth::user()->state_id);
		if($state){
        $request->user()->state_id = $state->name;
		}
        return response()->json(['success' => true, 'statusCode' => 200, 'user' => $request->user()]);
    }

    //change password
    public function changePassword(Request $request)
    {

        if (!(Hash::check($request->get('current-password'), Auth::user()->password))) {

            // The passwords matches
            return response()->json(['message' => 'Your current password does not matches with the password you provided. Please try again.']);
        }

        if (strcmp($request->get('current-password'), $request->get('new-password')) == 0) {
            //Current password and new password are same
            return response()->json(['message' => 'New Password cannot be same as your current password. Please choose a different password.']);
        }

        $validatedData = $request->validate([
            'current-password' => 'required',
            'new-password' => 'required|string|min:6|confirmed',
        ]);

        //Change Password
        $user = Auth::user();
        $user->password = bcrypt($request->get('new-password'));
        $user->save();

        return response()->json(['success' => true, 'statusCode' => 200, 'message' => "Password changed successfully !"]);

    }

    //edit the profile
    public function update(Request $request)
    {
        $user = User::find(Auth::user()->id);
        if ($request->has('name')) {
            $user->name = $request->name;
        }
        if ($request->has('email')) {
            $user->email = $request->email;
        }
        if ($request->has('state_id')) {
            $user->state_id = $request->state_id;
        }
        if ($request->has('city')) {
            $user->city = $request->city;
        }
        if ($request->has('address')) {
            $user->address = $request->address;
        }
        if ($request->has('phone')) {
            $user->phone = $request->phone;
        }
        if ($request->has('pincode')) {
            $user->pincode = $request->pincode;
        }

        $user->save();
        return response()->json(['success' => true, 'statusCode' => 200, 'message' => 'Profile Edited', 'user' => $user]);
    }

    //my orders
    public function myOrder()
    {
        $orders = Order::where('user_id', Auth::user()->id)->get();
        return response()->json([
            'message' => 'My Orders', 'orders' => $orders, 'success' => true, 'statusCode' => 200
        ]);
    }

    //add to wallet
    public function addWalletMoney(Request $request)
    {
        $wallet = new Wallet;
        $wallet->user_id = Auth::user()->id;
        $wallet->amount = $request->amount;
        $wallet->status = "Active";
        $wallet->debit_credit_type = "PLUS";
        $wallet->payment_id = $request->razorpay_payment_id;
        $wallet->save();
        return response()->json(['message' => 'Successfully Added Money in Wallet', 'success' => true, 'statusCode' => 200, 'wallet' => $wallet]);

    }

    //view wallet
    public function viewWallet()
    {
        $user = Auth::user()->id;
        $wallets = Wallet::where('user_id', $user)->get();
        $pluswallet = Wallet::where('user_id', Auth::user()->id)
            ->where('status', 'Active')
            ->where('debit_credit_type', 'PLUS')->sum('amount');

        $minuswallet = Wallet::where('user_id', Auth::user()->id)
            ->where('status', 'Active')
            ->where('debit_credit_type', 'MINUS')->sum('amount');

        $totalWallet = $pluswallet - $minuswallet;
        return response()->json(['success' => true, 'statusCode' => 200, 'message' => 'Wallet', 'wallet' => $wallets, 'total' => $totalWallet]);
    }
	//get all states
	public function getAllStates(){
		$states=State::all();
		return response()->json(['success'=>true,'statusCode'=>200,'states'=>$states]);
	}
}


